<?php
// app/Http/Controllers/Admin/QuestionController.php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Question;
use App\Models\Option;
use App\Models\Subject;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
class QuestionController extends Controller
{
    public function index(Request $request)
    {
        $filters = [
            'q'          => trim((string)$request->get('q', '')),
            'subject_id' => $request->get('subject_id'),
            'grade'      => $request->get('grade'),
            'level'      => $request->get('level'),
        ];

        $query = Question::with('subject')->orderByDesc('created_at');

        if ($filters['q'] !== '') {
            $needle = '%'.mb_strtolower($filters['q']).'%';
            $query->whereRaw('LOWER(stem_html) LIKE ?', [$needle]);
        }
        if (!empty($filters['subject_id'])) $query->where('subject_id', (int)$filters['subject_id']);
        if (strlen((string)$filters['grade'])) $query->where('grade', $filters['grade']);
        if (strlen((string)$filters['level'])) $query->where('level', (int)$filters['level']);

        $questions = $query->paginate(20)->appends($filters);

        $subjects = Subject::orderBy('name')->get(['id','name']);
        $grades   = Question::select('grade')->whereNotNull('grade')->distinct()->orderBy('grade')->pluck('grade');
        $levels   = range(1, 9);

        return view('admin.questions.index', compact('questions','filters','subjects','grades','levels'));
    }

    // Return JSON for Edit modal
    public function show(Question $question)
    {
        $question->load(['subject','options' => fn($q) => $q->orderByRaw("FIELD(label,'A','B','C','D')")]);
        return response()->json([
            'id' => $question->id,
            'subject_id' => $question->subject_id,
            'grade' => $question->grade,
            'level' => $question->level,
            'question_language' => $question->question_language,
            'b' => $question->b,
            'is_active' => (bool)$question->is_active,
            'stem_html' => $question->stem_html,
            'options' => $question->options->map(fn($o)=>[
                'id'=>$o->id,'label'=>$o->label,'text_html'=>$o->text_html,'weight'=>$o->weight,
            ])->values(),
        ]);
    }

    public function store(Request $request)
    {
        $data = $this->validatePayload($request);
        DB::transaction(function () use ($data) {
            $q = Question::create([
                'subject_id'        => $data['subject_id'],
                'grade'             => $data['grade'] ?? null,
                'level'             => $data['level'],
                'question_language' => $data['question_language'],
                'stem_html'         => $data['stem_html'],
                'b'                 => $data['b'] ?? null,
                'is_active'         => $data['is_active'] ?? true,
            ]);
            foreach (['A','B','C','D'] as $L) {
                $opt = $data['options'][$L] ?? ['text_html'=>'','weight'=>0];
                Option::create([
                    'question_id' => $q->id,
                    'label'       => $L,
                    'text_html'   => $opt['text_html'] ?? '',
                    'weight'      => $opt['weight'] ?? 0,
                ]);
            }
        });

        return back()->with('status', __('app.admin.questions.flash.created'));
    }

    public function update(Request $request, Question $question)
    {
        $data = $this->validatePayload($request, editing: true);
        DB::transaction(function () use ($question, $data) {
            $question->update([
                'subject_id'        => $data['subject_id'],
                'grade'             => $data['grade'] ?? null,
                'level'             => $data['level'],
                'question_language' => $data['question_language'],
                'stem_html'         => $data['stem_html'],
                'b'                 => $data['b'] ?? null,
                'is_active'         => $data['is_active'] ?? true,
            ]);

            // Upsert A..D by label
            $existing = $question->options()->get()->keyBy('label');
            foreach (['A','B','C','D'] as $L) {
                $payload = $data['options'][$L] ?? ['text_html'=>'','weight'=>0];
                $row = $existing->get($L);
                if ($row) {
                    $row->update([
                        'text_html' => $payload['text_html'] ?? '',
                        'weight'    => $payload['weight'] ?? 0,
                    ]);
                } else {
                    Option::create([
                        'question_id' => $question->id,
                        'label'       => $L,
                        'text_html'   => $payload['text_html'] ?? '',
                        'weight'      => $payload['weight'] ?? 0,
                    ]);
                }
            }
        });

        return back()->with('status', __('app.admin.questions.flash.updated'));
    }

    public function destroy(Question $question)
    {
        $question->delete(); // options cascade via FK
        return back()->with('status', __('app.admin.questions.flash.deleted'));
    }

    private function validatePayload(Request $request, bool $editing=false): array
    {
        $rules = [
            'subject_id'        => ['required','integer','exists:subjects,id'],
            'grade'             => ['nullable','string','max:50'],
            'level'             => ['required','integer','min:1','max:9'],
            'question_language' => ['required','in:en,ar'],
            'stem_html'         => ['required','string'],
            'b'                 => ['nullable','numeric'],
            'is_active' => ['required','boolean'], // was sometimes|boolean

            // Options A..D
            'options.A.text_html' => ['required','string'],
            'options.B.text_html' => ['required','string'],
            'options.C.text_html' => ['required','string'],
            'options.D.text_html' => ['required','string'],
            'options.A.weight'    => ['required','numeric','min:0','max:1'],
            'options.B.weight'    => ['required','numeric','min:0','max:1'],
            'options.C.weight'    => ['required','numeric','min:0','max:1'],
            'options.D.weight'    => ['required','numeric','min:0','max:1'],
        ];
        return $request->validate($rules);
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => ['required','file','mimes:xlsx,xls,csv','max:20480'],
        ]);

        $uploaded = $request->file('file');
        $path = $uploaded->getPathname();

        if (!\extension_loaded('zip') && preg_match('/\.xlsx$/i', $uploaded->getClientOriginalName())) {
            return back()->withErrors(['file' => __('app.admin.questions.import.need_zip')]);
        }

        @set_time_limit(120);
        @ini_set('memory_limit', '512M');

        $sheetsProcessed = 0;
        $rowsSeen = 0;
        $subjectsCreated = 0;
        $questionsCreated = 0;
        $questionsUpdated = 0;

        try {
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($path);

            foreach ($spreadsheet->getWorksheetIterator() as $sheet) {
                $sheetsProcessed++;

                $highestRow = (int) $sheet->getHighestRow();
                $highestCol = (string) $sheet->getHighestColumn();
                $highestColIndex = Coordinate::columnIndexFromString($highestCol);

                // Build normalized header map: [colIndex => header_key]
                $headers = [];
                for ($col = 1; $col <= $highestColIndex; $col++) {
                    $addr = Coordinate::stringFromColumnIndex($col) . '1';
                    $hVal = (string) $sheet->getCell($addr)->getValue();
                    $headers[$col] = mb_strtolower(preg_replace('/\s+/', '_', trim($hVal)));
                }

                // Helper to read a row value by header aliases
                $get = function(int $row, array $keys) use ($sheet, $headers, $highestColIndex) {
                    for ($col = 1; $col <= $highestColIndex; $col++) {
                        if (in_array($headers[$col] ?? '', $keys, true)) {
                            $addr = Coordinate::stringFromColumnIndex($col) . $row;
                            // getCalculatedValue handles formulas gracefully
                            $val = $sheet->getCell($addr)->getCalculatedValue();
                            return is_null($val) ? '' : trim((string)$val);
                        }
                    }
                    return '';
                };

                for ($row = 2; $row <= $highestRow; $row++) {
                    $rowsSeen++;

                    $subjectName  = $get($row, ['subject','subject_name','question_subject','subjectid','subject_id']);
                    $subjectIdRaw = $get($row, ['subject_id']);
                    $grade        = $get($row, ['grade','class','year']);
                    $level        = (int) ($get($row, ['level','difficulty','difficulty_level']) ?: 0);
                    $lang         = strtolower($get($row, ['question_language','language','lang']) ?: 'en');
                    $stem         = (string) $get($row, ['question','question_text','stem','text','stem_html']);

                    $opt = [
                        'A' => $get($row, ['option_a','a_text','option_1','option1']),
                        'B' => $get($row, ['option_b','b_text','option_2','option2']),
                        'C' => $get($row, ['option_c','c_text','option_3','option3']),
                        'D' => $get($row, ['option_d','d_text','option_4','option4']),
                    ];
                    $w = [
                        'A' => (float) ($get($row, ['weight_a','a_weight','option_1_weight','w1']) ?: 0),
                        'B' => (float) ($get($row, ['weight_b','b_weight','option_2_weight','w2']) ?: 0),
                        'C' => (float) ($get($row, ['weight_c','c_weight','option_3_weight','w3']) ?: 0),
                        'D' => (float) ($get($row, ['weight_d','d_weight','option_4_weight','w4']) ?: 0),
                    ];
                    $b = $get($row, ['rasch_b','b','rasch']);

                    if ($stem === '' || $level < 1 || $level > 9) {
                        continue;
                    }

                    // Resolve subject
                    if (is_numeric($subjectIdRaw)) {
                        $subjectId = (int)$subjectIdRaw;
                    } elseif ($subjectName !== '') {
                        $existing = \App\Models\Subject::where('name', $subjectName)->first();
                        if (!$existing) {
                            $existing = \App\Models\Subject::create(['name' => $subjectName]);
                            $subjectsCreated++;
                        }
                        $subjectId = $existing->id;
                    } else {
                        $existing = \App\Models\Subject::firstOrCreate(['name' => 'General']);
                        $subjectId = $existing->id;
                    }

                    // Strong match
                    $candidate = \App\Models\Question::where('subject_id', $subjectId)
                        ->where('question_language', $lang)
                        ->where('grade', $grade ?: null)
                        ->where('level', $level)
                        ->where('stem_html', $stem)
                        ->first();

                    \DB::transaction(function () use (
                        $candidate, $subjectId, $grade, $level, $lang, $stem, $b, $opt, $w,
                        &$questionsCreated, &$questionsUpdated
                    ) {
                        if (!$candidate) {
                            $q = \App\Models\Question::create([
                                'subject_id'        => $subjectId,
                                'grade'             => $grade ?: null,
                                'level'             => $level,
                                'question_language' => $lang,
                                'stem_html'         => $stem,
                                'b'                 => ($b === '' ? null : (float)$b),
                                'is_active'         => true,
                            ]);
                            foreach (['A','B','C','D'] as $L) {
                                \App\Models\Option::create([
                                    'question_id' => $q->id,
                                    'label'       => $L,
                                    'text_html'   => $opt[$L] ?? '',
                                    'weight'      => $w[$L] ?? 0,
                                ]);
                            }
                            $questionsCreated++;
                        } else {
                            $dirty = false;
                            if ((string)$candidate->b !== (string)($b === '' ? null : (float)$b)) $dirty = true;

                            $opts = $candidate->options()->get()->keyBy('label');
                            foreach (['A','B','C','D'] as $L) {
                                $row = $opts->get($L);
                                $t = (string)($opt[$L] ?? '');
                                $ww = (float)($w[$L] ?? 0);
                                if (!$row || $row->text_html !== $t || (float)$row->weight !== $ww) {
                                    $dirty = true;
                                    break;
                                }
                            }

                            if ($dirty) {
                                $candidate->update([
                                    'b' => ($b === '' ? null : (float)$b),
                                ]);
                                $opts = $candidate->options()->get()->keyBy('label');
                                foreach (['A','B','C','D'] as $L) {
                                    $row = $opts->get($L);
                                    if ($row) {
                                        $row->update([
                                            'text_html' => (string)($opt[$L] ?? ''),
                                            'weight'    => (float)($w[$L] ?? 0),
                                        ]);
                                    } else {
                                        \App\Models\Option::create([
                                            'question_id' => $candidate->id,
                                            'label'       => $L,
                                            'text_html'   => (string)($opt[$L] ?? ''),
                                            'weight'      => (float)($w[$L] ?? 0),
                                        ]);
                                    }
                                }
                                $questionsUpdated++;
                            }
                        }
                    });
                }
            }

            return back()->with('status', __(
                'app.admin.questions.import.success',
                [
                    'sheets'   => $sheetsProcessed,
                    'rows'     => $rowsSeen,
                    'subjects' => $subjectsCreated,
                    'created'  => $questionsCreated,
                    'updated'  => $questionsUpdated
                ]
            ));
        } catch (\Throwable $e) {
            report($e);
            return back()->withErrors([
                'file' => __('app.admin.questions.import.failed_detail', ['msg' => $e->getMessage()])
            ]);
        }
    }


}
