<?php
// app/Http/Controllers/Admin/StudentController.php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\School;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class StudentController extends Controller
{
    public function index(Request $request)
    {
        $filters = [
            'name'    => trim((string)$request->get('name','')),
            'grade'   => trim((string)$request->get('grade','')),
            'section' => trim((string)$request->get('section','')),
        ];

        $q = User::with('school')
            ->where('role', '!=', 'admin')
            ->orderByDesc('created_at');

        if ($filters['name'] !== '') {
            $needle = '%'.mb_strtolower($filters['name']).'%';
            $q->whereRaw('LOWER(name) LIKE ? OR LOWER(email) LIKE ? OR LOWER(student_id) LIKE ?', [$needle,$needle,$needle]);
        }
        if ($filters['grade'] !== '') {
            $q->where('grade', $filters['grade']);
        }
        if ($filters['section'] !== '') {
            $q->where('section', $filters['section']);
        }

        $students = $q->paginate(20)->appends($filters);
        $schools  = School::orderBy('name')->get(['id','name']);

        return view('admin.students.index', compact('students','filters','schools'));
    }

    // For Edit modal (JSON)
    public function show(User $student)
    {
        if ($student->role === 'admin') abort(403);
        return response()->json([
            'id'                 => $student->id,
            'name'               => $student->name,
            'email'              => $student->email,
            'student_id'         => $student->student_id,
            'username'           => $student->username,
            'grade'              => $student->grade,
            'level'              => $student->level,
            'section'            => $student->section,
            'gender'             => $student->gender,
            'nationality'        => $student->nationality,
            'preferred_language' => $student->preferred_language,
            'school_id'          => $student->school_id,
        ]);
    }

    public function store(Request $request)
    {
        $data = $this->validated($request);

        // derive a password if none set: student_id or random
        $plain = $request->input('password') ?: ($data['student_id'] ?? Str::random(10));
        $data['password'] = Hash::make($plain);
        $data['role'] = 'student';

        User::create($data);

        return back()->with('status', __('app.admin.students.flash.created'));
        // If you want to show the generated password: add ->with('generated_password', $plain)
    }

    public function update(Request $request, User $student)
    {
        if ($student->role === 'admin') abort(403);

        $data = $this->validated($request, $student);

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->input('password'));
        } else {
            unset($data['password']);
        }

        // prevent privilege escalation
        $data['role'] = 'student';

        $student->update($data);

        return back()->with('status', __('app.admin.students.flash.updated'));
    }

    public function destroy(User $student)
    {
        if ($student->role === 'admin') abort(403);
        $student->delete();
        return back()->with('status', __('app.admin.students.flash.deleted'));
    }

    private function validated(Request $request, ?User $existing = null): array
    {
        $id = $existing?->id ?? 'NULL';
        return $request->validate([
            'name'               => ['required','string','max:255'],
            'email'              => ["nullable","email","max:255","unique:users,email,{$id}"],
            'student_id'         => ["nullable","string","max:255","unique:users,student_id,{$id}"],
            'username'           => ["nullable","string","max:255","unique:users,username,{$id}"],
            'grade'              => ['nullable','string','max:50'],
            'level'              => ['nullable','string','max:50'],
            'section'            => ['nullable','string','max:50'],
            'gender'             => ['nullable','string','max:50'],
            'nationality'        => ['nullable','string','max:100'],
            'preferred_language' => ['nullable','in:en,ar'],
            'school_id'          => ['nullable','integer','exists:schools,id'],
            'password'           => ['nullable','string','min:6'], // only used if provided
        ]);
    }
}
